/* Filename: "MSP430_SDQLIB.c" */

//******************************************************************************
// This file contains the functions necessary to implement SDQ communications.
// The code is written specifically for the MSP430 microcontroller using an I/O
// port pin for the 1-wire SDQ interface. A ~5K pullup resistor is used at Vcc
//
// Written by: Randy Wu
// Company: Texas Instruments, Inc.
//******************************************************************************

#include "MSP430_SDQLIB.h"
#include "SDQ_writeByte.h"

//******************************************************************************
// void SDQ_init(int usec)
//
// Description : Initializes the SDQ module as a whole
// Returns : None
// Resources used : DCO @ 8 MHz, TACCR0
//******************************************************************************
void SDQ_init(void)
{
  DCOCTL = CALDCO_8MHZ;                     // Set DCO for 1MHz using
  BCSCTL1 = CALBC1_8MHZ;                    // calibration data in Info Flash
  TACTL = (TASSEL_2 | MC_2 | TACLR);        // Use SMCLK, continuous mode
  P1SEL &= ~(P1_SDQ);                       // P1.x GPIO function
  SDQ_HIGH;                                 // Keep SDQ bus in IDLE state
  SDQ_OUTPUT;                               // P1.x SDQ output direction
}

//******************************************************************************
// void SDQ_delayUSEC(int usec)
//
// Description : Goes to sleep for the specified number of microseconds
// Returns : None			
// Notes : This only works when (usec > 5) AND (usec < 8193)
//         For a delay of 5 usec or less, use one _NOP per usec instead
//******************************************************************************
void SDQ_delayUSEC(unsigned int usec)
{
  TACCTL0 = CCIE;                           // TACCR0 interrupt enabled
  _EINT();
  TACCR0 += ((SMCLKFREQHZ/1000000)*usec);
  LPM0;
  _DINT();
  TACCTL0 &= ~CCIE;                         // TACCR0 interrupt disabled
}

//******************************************************************************
// void SDQ_reset(void)
//
// Description : Creates the Reset signal to initiate SDQ communication
// Arguments : None  										 
// Returns: None
//******************************************************************************
void SDQ_reset(void)
{
  unsigned int i;

  SDQ_HIGH;
  SDQ_OUTPUT;
  SDQ_LOW;
  for (i = 0; i < ((480/65)+1); i++)    // Reset time should be > 480 usec
  {
    DLYUSEC_65();
  }
  SDQ_HIGH;
}

//******************************************************************************
// unsigned char SDQ_detect(void)
//
// Description : Detects if a device responds to Reset signal
// Arguments : PresenceTimer - Sets timeout if no device present
//             InputData - Actual state of GPIO
//             GotPulse - States if a pulse was detected								  
// Returns : GotPulse
//******************************************************************************
unsigned char SDQ_detect(void)
{
  unsigned int PresenceTimer = 300;
  static volatile unsigned char InputData;
  static volatile unsigned char GotPulse = 0;

  SDQ_INPUT;
  while ( (PresenceTimer > 0) && (GotPulse == 0) )
  {
    InputData = SDQ_READ;                   // Capture state of the SDQ bus
    if (InputData == 0)                     // If SDQ bus got pulled low,
    {                                       // then the device responded
      GotPulse = 1;
    }
    else                                    // If SDQ bus is still high,
    {                                       // then the device did not respond
      GotPulse = 0;
      --PresenceTimer;                      // Decrease timeout counter
    }
  }
  SDQ_delayUSEC(200);                       // Delay before attempting command
  
  return GotPulse;                          // Return if device detected or not
}

//******************************************************************************
// unsigned char SDQ_readBit(void)
//
// Description : Receives the bit value returned by the SDQ slave
// Arguments : none
// Returns : State of the SDQ bus (0 or 1)
//******************************************************************************
unsigned char SDQ_readBit(void)
{
  static unsigned char InBit;
  unsigned int i;
	
  SDQ_HIGH;
  SDQ_OUTPUT;
  SDQ_LOW;
  SDQ_INPUT;
  for (i = 0; i < 12; i++)                  // The total delay time between SDQ
  {                                         // going LOW and the actual read of
    _NOP();                                 // the SDQ bus should be ~12 usec
  }
  InBit = SDQ_READ;                         // Capture state of the SDQ bus
  //SDQ_delayUSEC(65);                        // Wait for end of read bit cycle
  DLYUSEC_65();                             // Wait for end of read bit cycle
  SDQ_HIGH;
  SDQ_OUTPUT;
  
  if (InBit)
    return 1;                               // Return SDQ HIGH
  else
    return 0;                               // Return SDQ LOW
}

//******************************************************************************
// unsigned char SDQ_readByte(void)
//
// Description : Reads 8 bits on the SDQ line and returns the byte value.
// Arguments : Databyte - Byte value returned by SDQ slave
//	       MaskByte - Used to seperate each bit
//             i - Used for 8 time loop										  
// Returns : DataByte
//******************************************************************************
unsigned char SDQ_readByte(void)
{
  unsigned char data = 0x00;
  unsigned char mask, i;
	
  _DINT();
  
  for (i = 0; i < 8; i++)                   // Select one bit at a time
  {
    mask = SDQ_readBit();		    // Read One Bit
    mask <<= i;                             // Determine bit position in byte
    data = (data | mask);                   // Keep adding bits to form the byte
  }
  
  SDQ_delayUSEC(200);                       // Delay before attempting command
  _EINT();
  
  return data;                              // Return byte value read
}

// Timer A0 Interrupt Service Routine
#pragma vector=TIMERA0_VECTOR
__interrupt void Timer_A (void)
{
  LPM0_EXIT;                                // Exit LPM0 on RETI
}
